<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

try {
    // Validate required fields
    if (!isset($_POST['delivery_id']) || !isset($_POST['status'])) {
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        exit();
    }
    
    $delivery_id = (int)$_POST['delivery_id'];
    $status = $_POST['status'];
    
    // Validate status
    $valid_statuses = ['pending', 'in-transit', 'delivered', 'cancelled'];
    if (!in_array($status, $valid_statuses)) {
        echo json_encode(['success' => false, 'message' => 'Invalid status']);
        exit();
    }
    
    // Check if delivery exists
    $stmt = $pdo->prepare("SELECT id, delivery_number, status, quotation_id FROM deliveries WHERE id = ?");
    $stmt->execute([$delivery_id]);
    $delivery = $stmt->fetch();
    
    if (!$delivery) {
        echo json_encode(['success' => false, 'message' => 'Delivery not found']);
        exit();
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Update delivery status
        $updateFields = ['status = ?', 'updated_at = NOW()'];
        $params = [$status, $delivery_id];
        
        // If marking as delivered, set delivered_at and delivered_by
        if ($status === 'delivered') {
            $updateFields[] = 'delivered_at = NOW()';
            $updateFields[] = 'delivered_by = ?';
            $params = [$status, $_SESSION['user_id'], $delivery_id];
        }
        
        $sql = "UPDATE deliveries SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // If delivery is cancelled and it was from a quotation, revert quotation status
        if ($status === 'cancelled' && $delivery['quotation_id']) {
            $stmt = $pdo->prepare("UPDATE quotations SET status = 'active', updated_at = NOW() WHERE id = ?");
            $stmt->execute([$delivery['quotation_id']]);
        }
        
        // Log the action
        $action = "Updated delivery status to " . str_replace('-', ' ', $status);
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs (user_id, action, details, ip_address, created_at) 
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $_SESSION['user_id'],
            $action,
            "Delivery #{$delivery['delivery_number']} status changed to $status",
            $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ]);
        
        // Commit transaction
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => "Delivery #{$delivery['delivery_number']} status updated to " . str_replace('-', ' ', $status),
            'status' => $status
        ]);
        
    } catch (Exception $e) {
        // Rollback transaction on error
        $pdo->rollBack();
        throw $e;
    }
    
} catch (PDOException $e) {
    error_log("Database error in update_delivery_status.php: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Database error occurred. Please try again.'
    ]);
} catch (Exception $e) {
    error_log("Error in update_delivery_status.php: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'An error occurred. Please try again.'
    ]);
}
?> 